#!/usr/bin/env python3
"""
kverkom_test.py – Simulátor ERP a Banky pre projekt KVERKOM

Použitie:
  kverkom_test.py erpStatus [-d]
  kverkom_test.py bankStatus [-d]
  kverkom_test.py newTransaction [-d]
  kverkom_test.py getAllTransactions POKLADNICA [-d]
  kverkom_test.py wait VATSK [POKLADNICA [TRANSACTIONID]] [-d]
  kverkom_test.py notify IBAN AMOUNT END_TO_END_ID [-h HOST] [-d]
  kverkom_test.py erp VATSK POKLADNICA [-d]
"""

import argparse
import hashlib
import http.client as http_client
import json
import logging
import ssl
import sys
import uuid

import requests
from paho.mqtt import client as mqtt

# ====== Konštanty / konfigurácia ======
ERP_HOST_DEFAULT = "api-erp-i.kverkom.sk"
BANK_HOST_DEFAULT = "api-banka-i.kverkom.sk"
MQTT_HOST_DEFAULT = "mqtt-i.kverkom.sk"
MQTT_PORT_DEFAULT = 8883
TIMEOUT_S = 30

# cesty k lokalnym suborom s certifikami a klucmi
CLIENT_CERT = ("./cert/kverkom-int-client.pem", "./cert/kverkom-int-client.key")
CA_BUNDLE = "./cert/kverkom-ca-bundle.pem"


# ====== Pomocné funkcie (debug/logy, HTTP volania) ======
def enable_debug_logging() -> None:
    # nízkoúrovňové logy http.client + urllib3/requests
    http_client.HTTPConnection.debuglevel = 1
    logging.basicConfig(level=logging.DEBUG)
    logging.getLogger("urllib3").setLevel(logging.DEBUG)
    logging.getLogger("urllib3.connectionpool").setLevel(logging.DEBUG)


def http_get(url: str):
    return requests.get(url, cert=CLIENT_CERT, verify=CA_BUNDLE, timeout=TIMEOUT_S)


def http_post(url: str, payload, headers=None):
    return requests.post(url, json=payload, headers=headers, cert=CLIENT_CERT, verify=CA_BUNDLE, timeout=TIMEOUT_S)


def generate_data_integrity_hash(iban: str, amount: str, currency: str, end_to_end_id: str) -> str:
    """
    Vytvorí SHA-256 hexdigest zo stringu "iban|amount|currency|endToEndId".
    Vstupy očakávame už v kanonickej textovej podobe (napr. amount "667.00").
    """
    input_string = f"{iban}|{amount}|{currency}|{end_to_end_id}"
    return hashlib.sha256(input_string.encode("utf-8")).hexdigest()  # .upper() ak treba veľké písmená


def mqtt_subscribe(host: str, port: int, topic: str, cafile: str, certfile: str, keyfile: str, stop_on=None, debug: bool = False) -> None:
    """
    Ekvivalent:
      mosquitto_sub -h mqtt-i.kverkom.sk -p 8883 -v -t "VATSK-1234567890/#" \
        --cafile kverkom-ca-bundle.pem --cert kverkom-int-client.pem \
        --key kverkom-int-client.key -d
    """

    # --- Callbacks (napodobnenie -v a -d) ---
    def on_connect(client, userdata, connect_flags, reason_code, properties):
        print(f"[MQTT.CONNECT]")
        if reason_code == 0:
            client.subscribe(topic, qos=1)
        else:
            print(f"[MQTT.CONNECT] Failed, {reason_code=}")

    def on_message(client: mqtt.Client, userdata, msg: mqtt.MQTTMessage):
        # -v štýl: "<topic> <payload>"
        try:
            payload = msg.payload.decode("utf-8", errors="replace")
        except Exception:
            payload = repr(msg.payload)
        print(f"[MQTT.MESSAGE] {msg.topic} {payload}")

        # Stop condition - any payload
        if stop_on and stop_on(payload):
            print(f"[MQTT.STOP] Prijatá očakávaná správa – koniec")
            client.disconnect()  # spôsobí, že loop_forever sa vráti

    def on_subscribe(client: mqtt.Client, userdata, mid, granted_qos, properties=None):
        print(f"[MQTT.SUBSCRIBE] topic={topic} mid={mid} granted_qos={granted_qos}")

    def on_disconnect(client, userdata, disconnect_flags, reason_code, properties):
        print(f"[MQTT.DISCONNECT]")

    def on_log(client: mqtt.Client, userdata, level, buf):
        # „-d“ výpisy
        print(f"[MQTT.LOG {level}] {buf}")

    # --- MQTT klient ---
    client = mqtt.Client(protocol=mqtt.MQTTv311, callback_api_version=mqtt.CallbackAPIVersion.VERSION2)

    # TLS ekvivalent --cafile/--cert/--key
    client.tls_set(
        ca_certs=cafile,
        certfile=certfile,
        keyfile=keyfile,
        cert_reqs=ssl.CERT_REQUIRED,
        tls_version=ssl.PROTOCOL_TLS_CLIENT,  # Py 3.7+
        ciphers=None,
    )
    client.tls_insecure_set(False)  # neignorovať verifikáciu servera

    # Debug logy (-d)
    if debug:
        # paho interné logy
        client.on_log = on_log
        logging.basicConfig(level=logging.DEBUG)

    # Handlery
    client.on_connect = on_connect
    client.on_message = on_message
    client.on_subscribe = on_subscribe
    client.on_disconnect = on_disconnect

    # Auto-reconnect podobne ako mosquitto_sub
    client.reconnect_delay_set(min_delay=3, max_delay=30)

    # Pripojenie a nekonečná slučka
    client.connect(host, port, keepalive=60)
    client.loop_forever(retry_first_connection=True)


# ====== Handlery príkazov ======
def cmd_erp_status(args: argparse.Namespace) -> int:
    """
    kverkom_test.py erpStatus [-d]
    """
    base = f"https://{ERP_HOST_DEFAULT}"
    url = f"{base}/api/v1/status"
    try:
        r = http_get(url)
        r.raise_for_status()
        print(r.text)
        return 0
    except requests.RequestException as e:
        print(f"[erpStatus] HTTP error: {e}", file=sys.stderr)
        return 1


def cmd_bank_status(args: argparse.Namespace) -> int:
    base = f"https://{BANK_HOST_DEFAULT}"
    url = f"{base}/api/v1/status"
    try:
        r = http_get(url)
        r.raise_for_status()
        print(r.text)
        return 0
    except requests.RequestException as e:
        print(f"[bankStatus] HTTP error: {e}", file=sys.stderr)
        return 1


def cmd_generate_new_transaction_id(args: argparse.Namespace) -> int:
    """
    kverkom_test.py newTransaction [-d]
    """
    base = f"https://{ERP_HOST_DEFAULT}"
    url = f"{base}/api/v1/generateNewTransactionId"
    r = http_post(url, {})
    print("[newTransaction]", r.status_code, r.text)
    return 0


def cmd_get_all_transactions(args: argparse.Namespace) -> int:
    """
    kverkom_test.py getAllTransactions POKLADNICA [-d]
    """
    base = f"https://{ERP_HOST_DEFAULT}"
    pokl = args.pokladnica
    url = f"{base}/api/v1/getAllTransactions/{pokl}"
    r = http_get(url)
    print("[getAllTransactions]", r.status_code, r.text)
    return 0


def cmd_notify(args: argparse.Namespace) -> int:
    base = f"https://{BANK_HOST_DEFAULT}"
    url = f"{base}/api/v1/payments"

    currency = "EUR"  # podľa potreby upravte alebo spravte z toho argument; zatial je to konstanta
    dih = generate_data_integrity_hash(args.iban, args.amount, currency, args.end_to_end_id)

    payload = {
        "transactionStatus": "ACCC",  # konstanta, nemeni sa
        "transactionAmount": {"currency": currency, "amount": args.amount},
        "endToEndId": args.end_to_end_id,
        "dataIntegrityHash": dih,
        "creditorAccount": {"iban": args.iban},
        # "creditorName": "...",  # prípadne doplniť ako ďalší argument; optional
    }

    headers = {
        "X-Request-ID": str(uuid.uuid4()),
        # "Content-Type" is set automatically when using json=...
    }

    try:
        r = http_post(url, payload, headers)
        r.raise_for_status()
        print("[notify]", r.status_code, r.text)
        return 0
    except requests.RequestException as e:
        print(f"[notify] HTTP error: {e}", file=sys.stderr)
        return 1


def cmd_erp(args: argparse.Namespace) -> int:
    """
    kverkom_test.py erp VATSK POKLADNICA [-d]
    """
    vatsk = args.vatsk
    pokl = args.pokladnica

    base = f"https://{ERP_HOST_DEFAULT}"
    r = http_post(f"{base}/api/v1/generateNewTransactionId", {})
    print("[newTransaction]", r.status_code, r.text)
    transaction_id = r.json()["transaction_id"]

    def stop_on(payload) -> bool:
        """
        True <=> payload je JSON objekt a data["endToEndId"] == transaction_id.
        Ak nie je JSON alebo nemá požadovaný prvok/hodnotu, vráti False.
        """
        # Ak prišlo ako bytes, dekódujme (UTF-8; tolerujeme BOM)
        if isinstance(payload, (bytes, bytearray)):
            try:
                payload = bytes(payload).decode("utf-8-sig")
            except UnicodeDecodeError:
                return False
        if not isinstance(payload, str):
            return False
        try:
            data = json.loads(payload)
        except Exception:
            return False
        return isinstance(data, dict) and data.get("endToEndId") == transaction_id

    topic = f"{vatsk}/{pokl}/{transaction_id}"
    mqtt_subscribe(MQTT_HOST_DEFAULT, MQTT_PORT_DEFAULT, topic, CA_BUNDLE, CLIENT_CERT[0], CLIENT_CERT[1], stop_on, args.debug)
    return 0


def cmd_wait(args: argparse.Namespace) -> int:
    """
    kverkom_test.py wait VATSK [POKLADNICA [TRANSACTIONID]] [-d]
    """
    vatsk = args.vatsk
    pokl = args.pokladnica
    txid = args.transactionid

    topic = f"{vatsk}/{pokl}/{txid}" if txid else f"{vatsk}/{pokl}/#" if pokl else f"{vatsk}/#"
    mqtt_subscribe(MQTT_HOST_DEFAULT, MQTT_PORT_DEFAULT, topic, CA_BUNDLE, CLIENT_CERT[0], CLIENT_CERT[1], None, args.debug)
    return 0


# ====== Parser argumentov (hierarchický) ======
def build_parser() -> argparse.ArgumentParser:
    parser = argparse.ArgumentParser(
        prog="kverkom_test.py",
        description="Simulátor ERP a Banky pre projekt KVERKOM",
        add_help=True,
    )
    # parser.add_argument("--help", action="help", help="Zobraziť pomoc a skončiť")

    sub = parser.add_subparsers(dest="command", required=True)

    # Spoločné voľby pre subpríkazy
    common = argparse.ArgumentParser()
    common.add_argument("-d", "--debug", action="store_true", help="Zapnúť debug výpisy")
    # common.add_argument("--help", action="help", help="Zobraziť pomoc príkazu a skončiť")

    # status
    p_erp_status = sub.add_parser("erpStatus", parents=[common], add_help=False, help="Zistiť status ERP API",
                                  usage="kverkom_test.py erpStatus [-d]", )
    p_erp_status.set_defaults(func=cmd_erp_status)

    p_bank_status = sub.add_parser("bankStatus", parents=[common], add_help=False, help="Zistiť status Bank API",
                                   usage="kverkom_test.py bankStatus [-d]", )
    p_bank_status.set_defaults(func=cmd_bank_status)

    # newTransaction
    p_gntid = sub.add_parser("newTransaction", parents=[common], add_help=False, help="Vygenerovať nové TransactionId",
                             usage="kverkom_test.py newTransaction [-d]")
    p_gntid.set_defaults(func=cmd_generate_new_transaction_id)

    # getAllTransactions POKLADNICA
    p_getall = sub.add_parser("getAllTransactions", parents=[common], add_help=False, help="Získať všetky transakcie pre pokladnicu",
                              usage="kverkom_test.py getAllTransactions POKLADNICA [-d]")
    p_getall.add_argument("pokladnica", metavar="POKLADNICA", help="Identifikátor pokladnice, napr. POKLADNICA-88812345678900001")
    p_getall.set_defaults(func=cmd_get_all_transactions)

    # wait VATSK [POKLADNICA [TRANSACTIONID]]
    p_wait = sub.add_parser("wait", parents=[common], add_help=False, help="Čakať na stav transakcie",
                            usage="kverkom_test.py wait VATSK [POKLADNICA [TRANSACTIONID]] [-d]", )
    p_wait.add_argument("vatsk", metavar="VATSK", help="Identifikátor subjektu, napr. VATSK-1234567890")
    p_wait.add_argument("pokladnica", metavar="POKLADNICA", nargs="?",
                        help="(Voliteľné) identifikátor pokladnice, napr. POKLADNICA-88812345678900001")
    p_wait.add_argument("transactionid", metavar="TRANSACTIONID", nargs="?",
                        help="(Voliteľné) identifikátor transakcie; napr. QR-f5357fe7062940f3b7cf3227f7039a44")
    p_wait.set_defaults(func=cmd_wait)

    # notify
    p_notify = sub.add_parser("notify", add_help=False, parents=[common], help="Odoslať notifikáciu o platbe")
    # p_notify.add_argument("--help", action="help", help="Zobraziť pomoc k príkazu 'notify'")
    p_notify.add_argument("iban", help="IBAN príjemcu (napr. SK...)")
    p_notify.add_argument("amount", help='Suma ako text (napr. "667.00")')
    p_notify.add_argument("end_to_end_id", help="EndToEndId (napr. QR-a23...f0")
    p_notify.set_defaults(func=cmd_notify)

    # erp VATSK POKLADNICA
    p_erp = sub.add_parser("erp", parents=[common], add_help=False, help="Spustiť simuláciu platby na pokladnici",
                           usage="kverkom_test.py run VATSK POKLADNICA [-d]", )
    p_erp.add_argument("vatsk", metavar="VATSK", help="Identifikátor subjektu, napr. VATSK-1234567890")
    p_erp.add_argument("pokladnica", metavar="POKLADNICA", help="Identifikátor pokladnice, napr. POKLADNICA-88812345678900001")
    p_erp.set_defaults(func=cmd_erp)

    return parser


def main(argv: list[str] | None = None) -> int:
    parser = build_parser()
    args = parser.parse_args(argv)

    if getattr(args, "debug", False):
        enable_debug_logging()

    try:
        return args.func(args)
    except KeyboardInterrupt:
        print("User abort", file=sys.stderr)
        return 130  # bežný exit code pre Ctrl+C (128 + SIGINT=2)


if __name__ == "__main__":
    sys.exit(main())
